<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Subcategory;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class CategoryController extends Controller
{
    public function index()
    {
        $categories = Category::with('brand')->orderBy('id','desc')->paginate(10);
        return Inertia::render('Admin/Categories', ['categories' => $categories]);
    }

    /**
     * Generate a unique slug for the given name and optional brand.
     * If $ignoreId is provided, exclude that record (useful for updates).
     * Examples: "vegetable" or "vegetable-china".
     */
    private function makeUniqueSlug(string $name, $brandId = null, $ignoreId = null)
    {
        $base = Str::slug($name);

        if ($brandId) {
            $brand = Brand::find($brandId);
            $brandPart = $brand ? Str::slug($brand->name) : null;
        } else {
            $brandPart = null;
        }

        $candidateBase = $brandPart ? ($base . '-' . $brandPart) : $base;
        $slug = $candidateBase;
        $i = 1;

        while (Category::where('slug', $slug)->when($ignoreId, function ($q) use ($ignoreId) {
            return $q->where('id', '!=', $ignoreId);
        })->exists()) {
            $slug = $candidateBase . '-' . $i++;
        }

        return $slug;
    }

    public function create()
    {
        $brands = Brand::orderBy('name')->get();
        return Inertia::render('Admin/Categories/Create', [
            'brands' => $brands,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'brand_id' => 'nullable|exists:brands,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $categoryData = [
            'name' => $data['name'],
            'slug' => $this->makeUniqueSlug($data['name'], $data['brand_id'] ?? null),
            'brand_id' => $data['brand_id'] ?? null,
        ];

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time() . '_' . $image->getClientOriginalName();
            $path = $image->storeAs('categories', $filename, 'public');
            $categoryData['image'] = $path;
        }

        Category::create($categoryData);

        return redirect()->route('admin.categories.index')->with('success', 'Category created successfully.');
    }

    public function show($id)
    {
        $category = Category::findOrFail($id);
        return Inertia::render('Admin/Categories/Show', ['category' => $category]);
    }

    public function edit($id)
    {
        $category = Category::findOrFail($id);
        $brands = Brand::orderBy('name')->get();
        return Inertia::render('Admin/Categories/EditForm', [
            'category' => $category,
            'brands' => $brands,
        ]);
    }

    public function update(Request $request, $id)
    {
        $category = Category::findOrFail($id);

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'brand_id' => 'nullable|exists:brands,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $categoryData = [
            'name' => $data['name'],
            'slug' => $this->makeUniqueSlug($data['name'], $data['brand_id'] ?? null, $category->id),
            'brand_id' => $data['brand_id'] ?? null,
        ];

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($category->image && \Storage::disk('public')->exists($category->image)) {
                \Storage::disk('public')->delete($category->image);
            }
            
            $image = $request->file('image');
            $filename = time() . '_' . $image->getClientOriginalName();
            $path = $image->storeAs('categories', $filename, 'public');
            $categoryData['image'] = $path;
        }

        $category->update($categoryData);

        return redirect()->route('admin.categories.index')->with('success', 'Category updated successfully.');
    }

    public function destroy($id)
    {
        $category = Category::findOrFail($id);

        $subcats = Subcategory::where('category_id', $category->id)->pluck('name')->toArray();

        if (!empty($subcats)) {
            return redirect()->route('admin.categories.index')->with('error', 'Cannot delete category; related subcategories: ' . implode(', ', $subcats) . '.');
        }

        // Delete image if exists
        if ($category->image && \Storage::disk('public')->exists($category->image)) {
            \Storage::disk('public')->delete($category->image);
        }

        $category->delete();
        return redirect()->route('admin.categories.index')->with('success', 'Category deleted successfully.');
    }

    public function blockers($id)
    {
        $category = Category::findOrFail($id);
        $subcats = Subcategory::where('category_id', $category->id)->pluck('name')->toArray();

        $blocked = !empty($subcats);

        return response()->json([
            'blocked' => $blocked,
            'subcategories' => $subcats,
        ]);
    }

    public function categoriesByBrand($brandId)
    {
        $categories = Category::where('brand_id', $brandId)->orderBy('name')->get(['id','name']);
        return response()->json($categories);
    }
}
