<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Deal;
use App\Models\Product;
use Illuminate\Http\Request;
use Inertia\Inertia;

class DealController extends Controller
{
    public function index()
    {
        $deals = Deal::with('product')->latest()->paginate(10);
        return Inertia::render('Admin/Deals/Index', compact('deals'));
    }

    public function create()
    {
        return Inertia::render('Admin/Deals/Create');
    }

    public function searchProducts(Request $request)
    {
        $query = $request->get('q');
        $products = Product::where('name', 'like', "%{$query}%")
            ->select('id', 'name', 'cost_price')
            ->limit(10)
            ->get();
        return response()->json($products);
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'original_price' => 'required|numeric|min:0',
            'deal_price' => 'required|numeric|min:0',
            'total_quantity' => 'required|integer|min:1',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'status' => 'required|in:active,inactive'
        ]);

        Deal::create([
            'product_id' => $request->product_id,
            'title' => Product::find($request->product_id)->name . ' Deal',
            'description' => 'Special deal for ' . Product::find($request->product_id)->name,
            'original_price' => $request->original_price,
            'deal_price' => $request->deal_price,
            'available_quantity' => $request->total_quantity,
            'sold_quantity' => 0,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'is_active' => $request->status === 'active'
        ]);
        
        return redirect()->route('admin.deals.index');
    }

    public function edit(Deal $deal)
    {
        $deal->load('product');
        return Inertia::render('Admin/Deals/Edit', compact('deal'));
    }

    public function update(Request $request, Deal $deal)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'original_price' => 'required|numeric|min:0',
            'deal_price' => 'required|numeric|min:0',
            'total_quantity' => 'required|integer|min:1',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'status' => 'required|in:active,inactive'
        ]);

        $deal->update([
            'product_id' => $request->product_id,
            'title' => Product::find($request->product_id)->name . ' Deal',
            'description' => 'Special deal for ' . Product::find($request->product_id)->name,
            'original_price' => $request->original_price,
            'deal_price' => $request->deal_price,
            'available_quantity' => $request->total_quantity,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'is_active' => $request->status === 'active'
        ]);
        
        return redirect()->route('admin.deals.index');
    }

    public function destroy(Deal $deal)
    {
        $deal->delete();
        return redirect()->route('admin.deals.index');
    }
}
