<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Product;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Subcategory;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use App\Models\Unit;
use App\Models\StockHistory;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with(['category','brand','subcategory','unit']);
        
        // Search functionality
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('sku', 'like', '%' . $request->search . '%')
                  ->orWhere('barcode', 'like', '%' . $request->search . '%');
            });
        }
        
        // Filter by status
        if ($request->status) {
            $query->where('status', $request->status);
        }
        
        // Filter by stock status
        if ($request->stock_status) {
            $query->where('stock_status', $request->stock_status);
        }
        
        // Filter by brand
        if ($request->brand_id) {
            $query->where('brand_id', $request->brand_id);
        }
        
        // Filter by category
        if ($request->category_id) {
            $query->where('category_id', $request->category_id);
        }
        
        // Filter by featured
        if ($request->has('featured')) {
            $query->where('is_featured', true);
        }
        
        $products = $query->orderBy('id','desc')->paginate(10)->withQueryString();
        
        return Inertia::render('Admin/Products', [
            'products' => $products,
            'filters' => $request->only(['search', 'status', 'stock_status', 'brand_id', 'category_id', 'featured']),
            'brands' => Brand::orderBy('name')->get(['id', 'name']),
            'categories' => Category::orderBy('name')->get(['id', 'name'])
        ]);
    }

    public function create()
    {
        $brands = Brand::orderBy('name')->get();
        $categories = Category::orderBy('name')->get();
        $subcategories = Subcategory::orderBy('name')->get();
        $units = Unit::orderBy('name')->get();
        
        return Inertia::render('Admin/Products/Create', [
            'brands' => $brands,
            'categories' => $categories,
            'subcategories' => $subcategories,
            'units' => $units,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'required|string|max:255|unique:products,sku',
            'price' => 'required|numeric|min:0',
            'cost_price' => 'required|numeric|min:0',
            'brand_id' => 'required|exists:brands,id',
            'category_id' => 'required|exists:categories,id',
            'subcategory_id' => 'nullable|exists:subcategories,id',
            'unit_id' => 'required|exists:units,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            // New fields
            'stock_quantity' => 'required|integer|min:0',
            'low_stock_threshold' => 'required|integer|min:0',
            'status' => 'required|in:active,inactive,draft',
            'is_featured' => 'boolean',
            'visibility' => 'required|in:public,private,hidden',
            'description' => 'nullable|string',
            'short_description' => 'nullable|string',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string|max:255',
            'barcode' => 'nullable|string|max:255',
            'discount_price' => 'nullable|numeric|min:0',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'tags' => 'nullable|string',
        ]);

        $productData = [
            'name' => $data['name'],
            'slug' => Str::slug($data['name']),
            'sku' => $data['sku'],
            'price' => $data['price'],
            'cost_price' => $data['cost_price'],
            'brand_id' => $data['brand_id'],
            'category_id' => $data['category_id'],
            'subcategory_id' => $data['subcategory_id'] ?? null,
            'unit_id' => $data['unit_id'],
            // New fields
            'stock_quantity' => $data['stock_quantity'],
            'low_stock_threshold' => $data['low_stock_threshold'],
            'status' => $data['status'],
            'is_featured' => $data['is_featured'] ?? false,
            'visibility' => $data['visibility'],
            'description' => $data['description'],
            'short_description' => $data['short_description'],
            'weight' => $data['weight'],
            'dimensions' => $data['dimensions'],
            'barcode' => $data['barcode'],
            'discount_price' => $data['discount_price'],
            'tax_rate' => $data['tax_rate'] ?? 0,
            'meta_title' => $data['meta_title'],
            'meta_description' => $data['meta_description'],
            'tags' => $data['tags'],
        ];

        // Set stock status based on quantity
        if ($productData['stock_quantity'] <= 0) {
            $productData['stock_status'] = 'out_of_stock';
        } elseif ($productData['stock_quantity'] <= $productData['low_stock_threshold']) {
            $productData['stock_status'] = 'low_stock';
        } else {
            $productData['stock_status'] = 'in_stock';
        }

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time() . '_' . $image->getClientOriginalName();
            $path = $image->storeAs('products', $filename, 'public');
            $productData['image'] = $path;
        }

        Product::create($productData);

        return redirect()->route('admin.products.index')->with('success', 'Product created successfully.');
    }

    public function show($id)
    {
        $product = Product::with(['category','brand','subcategory','unit'])->findOrFail($id);
        return Inertia::render('Admin/Products/Show', ['product' => $product]);
    }

    public function edit($id)
    {
        $product = Product::findOrFail($id);
        $brands = Brand::orderBy('name')->get();
        $categories = Category::orderBy('name')->get();
        $subcategories = Subcategory::orderBy('name')->get();
        $units = Unit::orderBy('name')->get();
        
        return Inertia::render('Admin/Products/Edit', [
            'product' => $product,
            'brands' => $brands,
            'categories' => $categories,
            'subcategories' => $subcategories,
            'units' => $units,
        ]);
    }

    public function update(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'required|string|max:255|unique:products,sku,' . $id,
            'price' => 'required|numeric|min:0',
            'cost_price' => 'required|numeric|min:0',
            'brand_id' => 'required|exists:brands,id',
            'category_id' => 'required|exists:categories,id',
            'subcategory_id' => 'nullable|exists:subcategories,id',
            'unit_id' => 'required|exists:units,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            // Enhanced fields
            'stock_quantity' => 'required|integer|min:0',
            'low_stock_threshold' => 'required|integer|min:0',
            'status' => 'required|in:active,inactive,draft',
            'is_featured' => 'boolean',
            'visibility' => 'required|in:public,private,hidden',
            'description' => 'nullable|string',
            'short_description' => 'nullable|string',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string|max:255',
            'barcode' => 'nullable|string|max:255',
            'discount_price' => 'nullable|numeric|min:0',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'tags' => 'nullable|string',
        ]);

        $productData = [
            'name' => $data['name'],
            'slug' => Str::slug($data['name']),
            'sku' => $data['sku'],
            'price' => $data['price'],
            'cost_price' => $data['cost_price'],
            'brand_id' => $data['brand_id'],
            'category_id' => $data['category_id'],
            'subcategory_id' => $data['subcategory_id'] ?? null,
            'unit_id' => $data['unit_id'],
            // Enhanced fields
            'stock_quantity' => $data['stock_quantity'],
            'low_stock_threshold' => $data['low_stock_threshold'],
            'status' => $data['status'],
            'is_featured' => $data['is_featured'] ?? false,
            'visibility' => $data['visibility'],
            'description' => $data['description'],
            'short_description' => $data['short_description'],
            'weight' => $data['weight'],
            'dimensions' => $data['dimensions'],
            'barcode' => $data['barcode'],
            'discount_price' => $data['discount_price'],
            'tax_rate' => $data['tax_rate'] ?? 0,
            'meta_title' => $data['meta_title'],
            'meta_description' => $data['meta_description'],
            'tags' => $data['tags'],
        ];

        // Update stock status based on quantity
        if ($productData['stock_quantity'] <= 0) {
            $productData['stock_status'] = 'out_of_stock';
        } elseif ($productData['stock_quantity'] <= $productData['low_stock_threshold']) {
            $productData['stock_status'] = 'low_stock';
        } else {
            $productData['stock_status'] = 'in_stock';
        }

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($product->image && Storage::disk('public')->exists($product->image)) {
                Storage::disk('public')->delete($product->image);
            }
            
            $image = $request->file('image');
            $filename = time() . '_' . $image->getClientOriginalName();
            $path = $image->storeAs('products', $filename, 'public');
            $productData['image'] = $path;
        }

        $product->update($productData);

        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully.');
    }

    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $product->delete();
        return redirect()->route('admin.products.index')->with('success', 'Product deleted.');
    }

    public function addStock(Request $request, $id)
    {
        $product = Product::findOrFail($id);
        
        $request->validate([
            'quantity' => 'required|integer|min:1',
            'notes' => 'nullable|string|max:255'
        ]);
        
        $product->addStock($request->quantity, $request->notes);
        
        return back()->with('success', 'Stock added successfully.');
    }

    public function stockHistory()
    {
        $histories = StockHistory::with('product')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
            
        return Inertia::render('Admin/StockHistory', [
            'histories' => $histories
        ]);
    }
}
